package com.yunyinsign.api.sdk.util;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.yunyinsign.api.sdk.core.BaseListRequest;
import com.yunyinsign.api.sdk.core.BaseRequest;

/**
 * 请求类型工具类
 */
public class RequestTypeUtil {

    /**
     * 构建请求体 - 通用版本
     */
    public static String buildRequestBody(BaseRequest<?> request) {
        if (request == null) {
            return "{}";
        }
        // 处理列表请求的特殊序列化
        if (request instanceof BaseListRequest) {
            BaseListRequest<?, ?> listRequest = (BaseListRequest<?, ?>) request;
            return listRequest.toJson();
        }

        // 处理文件上传请求（这些请求通常不需要JSON序列化）
        if (request instanceof com.yunyinsign.api.sdk.core.BaseFileUploadRequest) {
            // 文件上传请求通常不通过JSON序列化主体
            // 而是通过multipart/form-data
            return "{}"; // 返回空对象或适当的内容
        }
        // 将请求对象转换为JSON字符串
        return JSON.toJSONString(request);
    }

    /**
     * 构建请求体 - 支持序列化特性
     */
    public static String buildRequestBody(BaseRequest<?> request, SerializerFeature... features) {
        return JSON.toJSONString(request, features);
    }

    /**
     * 检查请求类型
     */
    public static boolean isFileUploadRequest(BaseRequest<?> request) {
        return request instanceof com.yunyinsign.api.sdk.core.BaseFileUploadRequest;
    }

    /**
     * 检查是否是下载请求
     */
    public static boolean isDownloadRequest(BaseRequest<?> request) {
        return request instanceof com.yunyinsign.api.sdk.core.BaseDownloadRequest;
    }

    /**
     * 检查是否是分页请求
     */
    public static boolean isPageRequest(BaseRequest<?> request) {
        return request instanceof com.yunyinsign.api.sdk.core.BasePageRequest;
    }

    /**
     * 检查是否是列表请求
     */
    public static boolean isListRequest(BaseRequest<?> request) {
        return request instanceof com.yunyinsign.api.sdk.core.BaseListRequest;
    }

    /**
     * 获取请求的API路径
     */
    public static String getApiPath(BaseRequest<?> request) {
        return request.getApiPath();
    }

    /**
     * 获取请求的HTTP方法
     */
    public static String getHttpMethod(BaseRequest<?> request) {
        return request.getHttpMethod();
    }

    /**
     * 构建请求头信息
     */
    public static String buildRequestHeaders(BaseRequest<?> request) {
        StringBuilder headers = new StringBuilder();
        if (request.getHeaderUserId() != null) {
            headers.append("userId: ").append(request.getHeaderUserId()).append("\n");
        }
        if (request.getHeaderSource() != null) {
            headers.append("source: ").append(request.getHeaderSource()).append("\n");
        }
        return headers.toString();
    }

    /**
     * 验证请求参数
     */
    public static void validateRequest(BaseRequest<?> request) {
        if (request == null) {
            throw new IllegalArgumentException("请求对象不能为空");
        }

        String apiPath = request.getApiPath();
        if (apiPath == null || apiPath.trim().isEmpty()) {
            throw new IllegalArgumentException("API路径不能为空");
        }

        String httpMethod = request.getHttpMethod();
        if (httpMethod == null || httpMethod.trim().isEmpty()) {
            throw new IllegalArgumentException("HTTP方法不能为空");
        }
    }

    /**
     * 处理特殊类型的请求
     */
    public static String buildSpecialRequestBody(BaseRequest<?> request) {
        // 处理列表请求的特殊序列化
        if (request instanceof com.yunyinsign.api.sdk.core.BaseListRequest) {
            com.yunyinsign.api.sdk.core.BaseListRequest<?, ?> listRequest =
                    (com.yunyinsign.api.sdk.core.BaseListRequest<?, ?>) request;
            return listRequest.toJson();
        }

        // 默认使用FastJSON序列化
        return JSON.toJSONString(request);
    }
}
