package com.yunyinsign.api.sdk.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;

/**
 * @author dyl
 */
public class FileUtil {
    /**
     * 读取文件为字节数组
     * @param filePath 文件路径
     * @return 文件字节数组
     * @throws IOException 文件读取异常
     */
    public static byte[] readFileToBytes(String filePath) throws IOException {
        File file = new File(filePath);
        if (!file.exists() || !file.isFile()) {
            throw new IllegalArgumentException("文件不存在: " + filePath);
        }

        try (InputStream inputStream = new FileInputStream(file)) {
            byte[] buffer = new byte[(int) file.length()];
            int bytesRead = inputStream.read(buffer);
            if (bytesRead != file.length()) {
                throw new IOException("文件读取不完整");
            }
            return buffer;
        }
    }

    /**
     * 读取文件为字节数组（使用NIO）
     * @param filePath 文件路径
     * @return 文件字节数组
     * @throws IOException 文件读取异常
     */
    public static byte[] readFileToBytesNIO(String filePath) throws IOException {
        return Files.readAllBytes(Paths.get(filePath));
    }

    /**
     * 获取文件扩展名
     * @param fileName 文件名
     * @return 文件扩展名（不含点）
     */
    public static String getFileExtension(String fileName) {
        if (fileName == null || !fileName.contains(".")) {
            return "";
        }
        return fileName.substring(fileName.lastIndexOf(".") + 1);
    }

    /**
     * 获取不带扩展名的文件名
     * @param fileName 文件名
     * @return 不带扩展名的文件名
     */
    public static String getFileNameWithoutExtension(String fileName) {
        if (fileName == null) {
            return "";
        }
        int dotIndex = fileName.lastIndexOf(".");
        return dotIndex > 0 ? fileName.substring(0, dotIndex) : fileName;
    }

    /**
     * 验证文件大小
     * @param fileBytes 文件字节数组
     * @param maxSizeMB 最大大小（MB）
     * @return 是否有效
     */
    public static boolean validateFileSize(byte[] fileBytes, int maxSizeMB) {
        if (fileBytes == null) {
            return false;
        }
        long maxBytes = maxSizeMB * 1024L * 1024L;
        return fileBytes.length <= maxBytes;
    }
}
