package com.yunyinsign.api.sdk.core;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.yunyinsign.api.sdk.result.OAuth2AccessToken;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;

import java.io.IOException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Token管理器 - 自动获取并缓存Token
 */
public class TokenManager {
    private final String appId;
    private final String appSecret;
    private final String baseUrl;
    private final OkHttpClient httpClient;
    private final ReentrantLock lock = new ReentrantLock();
    private volatile TokenCache tokenCache;

    private static class TokenCache {
        final String accessToken;
        final Long corpId;
        final long expireTime;

        TokenCache(OAuth2AccessToken token) {
            this.accessToken = token.getAccessToken();
            this.corpId = token.getCorpId();
            // 提前5分钟过期，避免临界情况
            // 处理expiresIn（可能是毫秒或秒）
            Integer expiresIn = token.getExpiresIn();
            if (expiresIn == null) {
                expiresIn = 7200; // 默认2小时
            }

            // 假设expiresIn是秒，如果看起来是毫秒（大于100000），则转换为秒
            if (expiresIn > 100000) {
                expiresIn = expiresIn / 1000;
            }
            long expiresMillis = TimeUnit.SECONDS.toMillis(token.getExpiresIn());
            this.expireTime = System.currentTimeMillis() + expiresMillis - TimeUnit.MINUTES.toMillis(5);
        }

        boolean isValid() {
            return System.currentTimeMillis() < expireTime;
        }
    }

    public TokenManager(String appId, String appSecret, String baseUrl) {
        this.appId = appId;
        this.appSecret = appSecret;
        this.baseUrl = baseUrl.endsWith("/") ? baseUrl.substring(0, baseUrl.length() - 1) : baseUrl;
        this.httpClient = new OkHttpClient.Builder()
                .connectTimeout(10, TimeUnit.SECONDS)
                .readTimeout(30, TimeUnit.SECONDS)
                .writeTimeout(30, TimeUnit.SECONDS)
                .build();
    }

    /**
     * 获取Token
     */
    public String getAccessToken() {
        if (tokenCache == null || !tokenCache.isValid()) {
            refreshToken();
        }
        return tokenCache.accessToken;
    }

    /**
     * 获取企业ID
     */
    public Long getCorpId() {
        if (tokenCache == null || !tokenCache.isValid()) {
            refreshToken();
        }
        return tokenCache.corpId;
    }
    /**
     * 获取企业ID
     */
    public String getAppId() {
        if (tokenCache == null || !tokenCache.isValid()) {
            refreshToken();
        }
        return this.appId;
    }
    /**
     * 刷新Token - 使用锁避免并发重复获取
     */
    private void refreshToken() {
        lock.lock();
        try {
            // 双重检查锁
            if (tokenCache == null || !tokenCache.isValid()) {
                OAuth2AccessToken token = fetchTokenFromServer();
                tokenCache = new TokenCache(token);
            }
        } finally {
            lock.unlock();
        }
    }

    /**
     * 从服务器获取Token
     * 实际调用OAuth2接口获取访问令牌
     */
    private OAuth2AccessToken fetchTokenFromServer() {
        try {
            // 构建JSON请求体
            String json = String.format(
                    "{\"appId\":\"%s\",\"appSecret\":\"%s\"}",
                    appId, appSecret
            );

            RequestBody body = RequestBody.create(
                    json,
                    MediaType.parse("application/json")
            );

            Request request = new Request.Builder()
                    .url(baseUrl + "/service/getAccessToken")  // 根据实际接口路径
                    .post(body)
                    .addHeader("Content-Type", "application/json")
                    .addHeader("Accept", "application/json")
                    .build();

            Response response = httpClient.newCall(request).execute();

            if (!response.isSuccessful()) {
                throw new RuntimeException("获取Token失败: " + response.code() + " - " + response.message());
            }

            String responseBody = response.body().string();
            // 解析响应
            JSONObject jsonObject = JSON.parseObject(responseBody);
            String code = jsonObject.getString("code");
            if (!"200".equals(code)) {
                String msg = jsonObject.getString("msg");
                throw new RuntimeException("获取Token失败: " + code + " - " + msg);
            }

            JSONObject data = jsonObject.getJSONObject("data");
            if (data == null) {
                throw new RuntimeException("响应中没有data字段");
            }
            // 创建Token对象
            OAuth2AccessToken token = new OAuth2AccessToken();
            token.setAccessToken(data.getString("accessToken"));

            // 处理corpId，可能是字符串格式
            String corpIdStr = data.getString("corpId");
            if (corpIdStr != null) {
                try {
                    token.setCorpId(Long.parseLong(corpIdStr));
                } catch (NumberFormatException e) {
                    throw new RuntimeException("corpId格式错误: " + corpIdStr);
                }
            }

            // 处理expiresIn，可能是字符串格式
            String expiresInStr = data.getString("expiresIn");
            if (expiresInStr != null) {
                try {
                    // 如果看起来是毫秒（长度大于5位），则转换为秒
                    long expiresIn = Long.parseLong(expiresInStr);
                    if (expiresIn > 100000) {
                        token.setExpiresIn((int)(expiresIn / 1000));
                    } else {
                        token.setExpiresIn((int)expiresIn);
                    }
                } catch (NumberFormatException e) {
                    throw new RuntimeException("expiresIn格式错误: " + expiresInStr);
                }
            } else {
                token.setExpiresIn(7200); // 默认2小时
            }

            return token;

        } catch (IOException e) {
            throw new RuntimeException("获取Token网络异常: " + e.getMessage(), e);
        }
    }

    /**
     * 强制刷新Token（可用于手动刷新）
     */
    public void forceRefreshToken() {
        lock.lock();
        try {
            OAuth2AccessToken token = fetchTokenFromServer();
            tokenCache = new TokenCache(token);
        } finally {
            lock.unlock();
        }
    }

    /**
     * 清除Token缓存
     */
    public void clearToken() {
        tokenCache = null;
    }
}
