package com.yunyinsign.api.sdk.client;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.yunyinsign.api.sdk.core.*;
import com.yunyinsign.api.sdk.result.GetFileUploadRes;
import com.yunyinsign.api.sdk.util.RequestTypeUtil;
import okhttp3.*;

import java.io.IOException;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class YunYinClient {
    private final TokenManager tokenManager;
    private final OkHttpClient httpClient;
    private final String baseUrl;
    private final Long userId;

    /**
     * 构造函数
     * @param appId 客户端ID
     * @param appSecret 客户端密钥
     * @param baseUrl 基础URL
     */
    public YunYinClient(String appId, String appSecret, String baseUrl) {
        this.tokenManager = new TokenManager(appId, appSecret, baseUrl);
        this.baseUrl = baseUrl.endsWith("/") ? baseUrl.substring(0, baseUrl.length() - 1) : baseUrl;
        this.userId = null;
        this.httpClient = new OkHttpClient.Builder()
                .connectTimeout(10, TimeUnit.SECONDS)
                .readTimeout(30, TimeUnit.SECONDS)
                .writeTimeout(30, TimeUnit.SECONDS)
                .build();
    }

    /**
     * 构造函数
     * @param appId 客户端ID
     * @param appSecret 客户端密钥
     * @param baseUrl 基础URL
     */
    public YunYinClient(String appId, String appSecret, String baseUrl, Long userId) {
        this.tokenManager = new TokenManager(appId, appSecret, baseUrl);
        this.baseUrl = baseUrl.endsWith("/") ? baseUrl.substring(0, baseUrl.length() - 1) : baseUrl;
        this.userId = userId;
        this.httpClient = new OkHttpClient.Builder()
                .connectTimeout(10, TimeUnit.SECONDS)
                .readTimeout(30, TimeUnit.SECONDS)
                .writeTimeout(30, TimeUnit.SECONDS)
                .build();
    }

    /**
     * 执行API请求 - 主要方法，根据请求对象的泛型自动返回对应的响应类型
     * @param request 请求对象
     * @param <R> 响应类型
     * @return 响应对象
     */
    public <R extends BaseResponse> R execute(BaseRequest<R> request) {
        return executeRequest(request, request.getResponseType());
    }

    /**
     * 执行API请求 - 主要方法，根据请求对象的泛型自动返回对应的响应类型
     * @param request 请求对象
     * @param <R> 响应类型
     * @return 响应对象
     */
    public <R extends BaseResponse> R execute(BaseRequest<R> request, Long userId) {
        if(userId != null){
            request.setHeaderUserId(userId);
        }
        return executeRequest(request, request.getResponseType());
    }

    /**
     * 执行API请求 - 简化版本（返回原始JSON）
     * @param request 请求对象
     * @return JSON响应字符串
     */
    public String executeRaw(BaseRequest<?> request) {
        try {
            Headers headers = buildHeaders(request);
            String jsonBody = RequestTypeUtil.buildRequestBody(request);
            Request httpRequest = buildRequest(request, headers, jsonBody);
            Response response = httpClient.newCall(httpRequest).execute();
            if (!response.isSuccessful()) {
                throw new RuntimeException("HTTP请求失败: " + response.code() + " - " + response.message());
            }
            ResponseBody body = response.body();
            if (body == null) {
                return "";
            }
            return body.string();
        } catch (IOException e) {
            throw new RuntimeException("API调用异常", e);
        }
    }

    /**
     * 通用文件上传方法
     * @param request 文件上传请求（继承自BaseFileUploadRequest）
     * @return 文件上传响应
     */
    public <R extends BaseResponse> R uploadFile(BaseFileUploadRequest<R> request) {
        return uploadFile(request, request.getResponseType());
    }
    public <R extends BaseResponse> R uploadFile(BaseFileUploadRequest<R> request, Long userId) {
        if(userId != null){
            request.setHeaderUserId(userId);
        }
        return uploadFile(request, request.getResponseType());
    }

    /**
     * 导出Excel文件 - 通用方法
     * @param request 导出请求
     * @return 导出的Excel文件
     */
    public DownloadResponse export(BaseDownloadRequest<?> request) {
        return downloadRequest(request);
    }
    public DownloadResponse export(BaseDownloadRequest<?> request, Long userId) {
        if(userId != null){
            request.setHeaderUserId(userId);
        }
        return downloadRequest(request);
    }
    /**
     * 下载方法 - 直接返回字节数组和文件名
     * @param request 下载请求
     * @return 下载响应（包含字节数组和文件名）
     */
    public DownloadResponse download(BaseDownloadRequest<?> request) {
        return downloadRequest(request);
    }
    public DownloadResponse download(BaseDownloadRequest<?> request, Long userId) {
        if(userId != null){
            request.setHeaderUserId(userId);
        }
        return downloadRequest(request);
    }
    /**
     * 通用文件上传方法（指定响应类型）
     * @param request 文件上传请求
     * @param responseClass 响应类
     * @return 响应对象
     */
    private <R extends BaseResponse> R uploadFile(BaseFileUploadRequest<R> request, Class<R> responseClass)  {
        Response response = null;
        try {
            // 准备请求参数
            String url = baseUrl + request.getApiPath();
            String accessToken = tokenManager.getAccessToken();
            String appId = tokenManager.getAppId();

            // 验证文件
            if (request.getFileBytes() == null || request.getFileBytes().length == 0) {
                throw new IllegalArgumentException("文件内容不能为空");
            }

            String fileName = request.getFileName();
            if (fileName == null || fileName.trim().isEmpty()) {
                throw new IllegalArgumentException("文件名不能为空");
            }

            // 构建Multipart请求体
            MultipartBody.Builder multipartBuilder = new MultipartBody.Builder()
                    .setType(MultipartBody.FORM);

            // 添加文件部分
            String fieldName = request.getFieldName();
            multipartBuilder.addFormDataPart(fieldName, fileName,
                    RequestBody.create(
                            request.getFileBytes(),
                            MediaType.parse("application/octet-stream")
                    )
            );

            // 添加额外的表单参数
            Map<String, String> formParams = request.getFormParams();
            if (formParams != null && !formParams.isEmpty()) {
                for (Map.Entry<String, String> entry : formParams.entrySet()) {
                    multipartBuilder.addFormDataPart(entry.getKey(), entry.getValue());
                }
            }
            RequestBody body = multipartBuilder.build();
            Headers.Builder headerBuilder = new Headers.Builder()
                    .add("accessToken", accessToken)
                    .add("source", "pc")
                    .add("userId", String.valueOf(userId))
                    .add("appId", appId);

            if (request.getHeaderUserId() != null) {
                headerBuilder.add("userId", String.valueOf(request.getHeaderUserId()));
            }
            if (request.getHeaderSource() != null) {
                headerBuilder.add("source", request.getHeaderSource());
            }
            // 构建请求头
            Headers headers = headerBuilder.build();

            // 构建HTTP请求
            Request httpRequest = new Request.Builder()
                    .url(url)
                    .headers(headers)
                    .post(body)
                    .build();

            // 执行请求
            response = httpClient.newCall(httpRequest).execute();

            if (!response.isSuccessful()) {
                String errorBody = response.body() != null ? response.body().string() : "";
                throw new RuntimeException("文件上传失败: " + response.code() + " - " + response.message() +
                        "\n响应内容: " + errorBody);
            }

            // 解析响应
            String responseBody = response.body().string();

            // 解析JSON
            JSONObject jsonObject = JSON.parseObject(responseBody);
            String code = jsonObject.getString("code");
            String msg = jsonObject.getString("msg");

            if (!"200".equals(code)) {
                throw new RuntimeException("文件上传失败: " + code + " - " + msg);
            }

            // 将整个响应体解析为具体的响应类
            R result = JSON.parseObject(responseBody, responseClass);
            // 确保基础字段被正确设置
            result.setCode(code);
            result.setMsg(msg);

            return result;
        } catch (IOException e) {
            throw new RuntimeException("文件上传网络异常", e);
        } catch (Exception e) {
            throw new RuntimeException("文件上传失败: " + e.getMessage(), e);
        } finally {
            // 确保关闭response，防止资源泄漏
            if (response != null) {
                response.close();
            }
        }
    }

    /**
     * 下载方法 - 直接返回字节数组和文件名
     * @param request 下载请求
     * @return 下载响应（包含字节数组和文件名）
     */
    public DownloadResponse downloadRequest(BaseDownloadRequest<?> request) {
        Response response = null;
        try {
            // 构建请求头
            Headers headers = buildHeaders(request);
            String jsonBody = JSON.toJSONString(request);
            // 构建请求
            Request httpRequest = buildRequest(request, headers, jsonBody);
            // 执行请求
            response = httpClient.newCall(httpRequest).execute();

            // 检查响应
            if (!response.isSuccessful()) {
                throw new RuntimeException("下载失败: " + response.code() + " - " + response.message());
            }

            // 检查是否是错误响应（application/json）
            String contentType = response.header("Content-Type", "");
            if (contentType.contains("application/json")) {
                String responseBody = response.body().string();
                try {
                    // 尝试解析为JSON错误响应
                    JSONObject json = JSON.parseObject(responseBody);
                    String code = json.getString("code");
                    String msg = json.getString("msg");
                    if (!"200".equals(code)) {
                        throw new RuntimeException("下载失败: " + code + " - " + msg);
                    }
                } catch (Exception e) {
                    // 不是预期的JSON格式，继续处理
                }
            }

            // 获取文件流
            ResponseBody responseBody = response.body();
            if (responseBody == null) {
                throw new RuntimeException("响应体为空");
            }

            // 读取字节数组
            byte[] fileBytes = responseBody.bytes();

            // 提取文件名
            String fileName = extractFileName(response);

            // 获取Content-Type
            String fileContentType = response.header("Content-Type", "application/octet-stream");

            return new DownloadResponse(fileBytes, fileName, fileContentType);

        } catch (IOException e) {
            throw new RuntimeException("下载异常: " + e.getMessage(), e);
        } finally {
            if (response != null) {
                response.close();
            }
        }
    }

    /**
     * 内部执行方法 - 支持具体的响应类
     */
    private <R extends BaseResponse> R executeRequest(BaseRequest<R> request, Class<R> responseClass) {
        Response httpResponse = null;
        try {
            // 准备请求参数
            Headers headers = buildHeaders(request);
            String jsonBody = RequestTypeUtil.buildRequestBody(request);
            Request httpRequest = buildRequest(request, headers, jsonBody);

            // 执行请求
            httpResponse = httpClient.newCall(httpRequest).execute();
            if (!httpResponse.isSuccessful()) {
                throw new RuntimeException("HTTP请求失败: " + httpResponse.code() + " - " + httpResponse.message());
            }

            // 解析响应
            String responseBody = httpResponse.body().string();
            JSONObject jsonObject = JSON.parseObject(responseBody);
            String code = jsonObject.getString("code");
            String msg = jsonObject.getString("msg");

            if (!"200".equals(code)) {
                throw new RuntimeException("API调用失败: " + code + " - " + msg);
            }

            // 将响应体解析为具体的响应类
            R result = JSON.parseObject(responseBody, responseClass);

            // 确保基础字段被正确设置
            result.setCode(code);
            result.setMsg(msg);

            // 处理分页响应
            if (result instanceof BasePageResponse) {
                BasePageResponse pageResponse = (BasePageResponse) result;

                // 设置分页信息
                pageResponse.setListPageNo(jsonObject.getInteger("listPageNo"));
                pageResponse.setCountInPage(jsonObject.getInteger("countInPage"));
                pageResponse.setListPageCount(jsonObject.getInteger("listPageCount"));
                pageResponse.setTotalCount(jsonObject.getInteger("totalCount"));

                // 确保data字段被正确设置
                // 由于具体的响应类中已经定义了data字段，并且FastJSON会自动映射，这里不需要额外处理
            }

            return result;

        } catch (IOException e) {
            throw new RuntimeException("API调用异常", e);
        } catch (Exception e) {
            throw new RuntimeException("解析响应失败: " + e.getMessage(), e);
        } finally {
            // 确保关闭response，防止资源泄漏
            if (httpResponse != null) {
                httpResponse.close();
            }
        }
    }


    /**
     * 从响应头提取文件名
     */
    private String extractFileName(Response response) {
        String contentDisposition = response.header("Content-Disposition");
        if (contentDisposition != null) {
            // 匹配 filename="文件名" 或 filename*=UTF-8''文件名
            Pattern pattern = Pattern.compile("filename\\*?=\"?([^\";]+)");
            Matcher matcher = pattern.matcher(contentDisposition);
            if (matcher.find()) {
                String fileName = matcher.group(1);

                // 如果是URL编码的，进行解码
                if (fileName.contains("%")) {
                    try {
                        fileName = URLDecoder.decode(fileName, StandardCharsets.UTF_8.name());
                    } catch (Exception e) {
                        // 解码失败，使用原始文件名
                    }
                }
                return fileName;
            }
        }
        // 如果没有找到，返回默认文件名
        return "download_file";
    }

    private Headers buildHeaders(BaseRequest<?> request) {
        String accessToken = tokenManager.getAccessToken();
        String appId = tokenManager.getAppId();

        // 构建请求头
        Headers.Builder headerBuilder = new Headers.Builder()
                .add("accessToken", accessToken)
                .add("appId", appId)
                .add("source", "pc")
                .add("userId", String.valueOf(userId))
                .add("Content-Type", "application/json");

        if (request.getHeaderUserId() != null) {
            headerBuilder.add("userId", String.valueOf(request.getHeaderUserId()));
        }
        if (request.getHeaderSource() != null) {
            headerBuilder.add("source", request.getHeaderSource());
        }
        return headerBuilder.build();
    }

    private Request buildRequest(BaseRequest<?> request, Headers headers, String jsonBody) {
        RequestBody body = RequestBody.create(
                jsonBody,
                MediaType.parse("application/json")
        );
        // 构建请求
        return new Request.Builder()
                .url(baseUrl + request.getApiPath())
                .headers(headers)
                .method(request.getHttpMethod(), body)
                .build();
    }
}
